
/*******************************************************************************
*
*	Copyright (c) 2024 skipper
*	http://www.heroesden.link
*
*	Terms of usage: Light license agreement v1.
*
*******************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/*******************************************************************************
*
*	Function prototypes.
*
*******************************************************************************/


void convertComments(void);
void readSrcFile(const char *path);
void writeModFile(void);


/*******************************************************************************
*
*	Global variables.
*
*******************************************************************************/


struct
{
	//	Pointer to dynamically allocated memory which contains whole source
	//	file's code and has extra free space for code conversions.
	//
	char *buffer;

	//	Size of the allocated buffer. It is calculated as SIZEOF(SRC_FILE)*2.
	//
	size_t bufSize;

	//	Filesystem path of converted file with ".c" extension. In Windows API
	//	the maximum length for a path is 260 characters.
	//
	char dstPath[260];

	//	Sequential number of the source file in the list of command line
	//	arguments.
	//
	size_t index;

	//	Filesystem path of source file with ".sal" extension. It taken from
	//	command line arguments.
	//
	const char *path;

	//	Size of unmodified source file.
	//
	size_t size;
}
srcFile;


/*******************************************************************************
*
*	Function declarations.
*
*******************************************************************************/


int main(int argc, char *argv[])
{
	int i;


	/***************************************************************************
	*
	*	Open a source file.
	*
	***************************************************************************/


	//	Does the user specified a file for conversion?
	//
	if(argc < 2)
	{
		printf("Error. No input files specified.");
		return 0;
	}

	//	Handle source files from command line arguments.
	//
	for(i = 1; i < argc; i++)
	{
		memset(&srcFile, 0, sizeof(srcFile));
		srcFile.index = i;
		readSrcFile(argv[i]);

		convertComments();

		writeModFile();
		free(srcFile.buffer);
	}

	printf("Congrutalations! All is fine.");
	return 0;
}


//	Change all comments to multi-line style.
//
void convertComments(void)
{
	size_t i;

	char *ch;
	char *comEnd;
	char *comStart;
	char *tabStart;

	char isEndingCom = 0;
	char isStartingCom = 1;
	char isComBlank;

	ch = srcFile.buffer;

	while(*ch != '\0')
	{
		//	End of a single-line comment is found.
		//
		//if(*ch == '\n' && comStart != NULL)
		//{
		//	break;
		//}

		//	An ordinary character is found.
		//
		if(*ch != '/' || *(ch + 1) != '/')
		{
			ch++;
			continue;
		}

		//	Single-line comment is found.
		//
		comStart = ch;
		isComBlank = 1;

		//	Go back and find tabs and spaces.
		//
		tabStart = NULL;

		while(ch > 0 && ch > srcFile.buffer)
		{
			ch--;

			//	Tab or Space is found.
			//
			if(*ch == '\t' || *ch == ' ')
			{
				tabStart = ch;
				continue;
			}

			//	The comment is starting not at the beginning of the line but
			//	after some code. For example,
			//
			//	MyMegaFunc(a, b)
			//	{
			//		return a + b;	//	This is very useful function.
			//	}
			//
			if(*ch != '\n')
			{
				isEndingCom = 1;
				break;
			}

			//	Printable character is found.
			//
			break;
		}

		//	Go forward and find end of the comment line. Also determine if
		//	the comment is blank or not.
		//
		ch = comStart + 2;
		while(*ch != '\n' && *ch != '\0')
		{
			if(*ch != ' ' && *ch != '\t') isComBlank = 0;

			ch++;
		}
		comEnd = ch;

		//	Go forward and find next comment with the same indent.
		//
		if(isEndingCom == 0)
		{
			ch++;

			if(tabStart == NULL && (*ch != '/' || *(ch + 1) != '/'))
			{
				isEndingCom = 1;
			}

			if(tabStart != NULL)
			{
				int result;

				result = strncmp(ch, tabStart, comStart - tabStart + 2);

				if(result != 0)
				{
					isEndingCom = 1;
				}
			}
		}

		//	Change opening comment delimiter.
		//
		if(isStartingCom == 1)
		{
			//	The comment is the first in the comment block (change "//" to
			//	"/*").
			//
			*(comStart + 1) = '*';
		}
		else if(isStartingCom == 0 && isEndingCom == 1 && isComBlank == 1)
		{
			//	The comment is blank and last in the comment block
			//	(change "//" to "*/").
			//
			*comStart = '*';
		}
		else
		{
			//	The comment is placed in the middle or ending of the comment
			//	block (change "//" to " *").
			//
			*comStart = ' ';
			*(comStart + 1) = '*';
		}

		//	Add closing comment delimiter.
		//
		if(isEndingCom == 1)
		{
			if(isStartingCom == 1 || isComBlank == 0)
			{
				ch = comEnd + 3;
	
				//	Add free space just after comment. It is needed for closing
				//	comment delimiter.
				//
				memmove(ch, comEnd, strlen(comEnd) + 1);
	
				//	Add " */" to the end of the comment.
				//
				memcpy(comEnd, " */", 3);
			}
			else
			{
				ch = comEnd;
			}

			//	Settings for next comment.
			//
			isStartingCom = 1;
			isEndingCom = 0;
		}
		else
		{
			//	The comment have continuation on next line.
			//
			isStartingCom = 0;
		}
	}
}


//	Read the content of the source file and place it in the buffer.
//
void readSrcFile(const char *path)
{
	//	Pointer to the source file.
	//
	FILE *fileObj;


	//	Isn't path of source file too long?
	//
	{
		size_t pathLen = strlen(path);

		if(pathLen >= sizeof(srcFile.dstPath))
		{
			printf("Error in file #%d. Length of source file path is too big: %d "
				"characters.", srcFile.index, pathLen);

			exit(EXIT_FAILURE);
		}
	}

	//	Copy source file path without '.sal' extension to destination file path
	//	and append '.c' to the end.
	//
	{
		char *dot = strrchr(path, '.');
		size_t dotPos;

		if(dot == NULL)
		{
			printf("Error in file #%d. Can't find dot in source file path: %s",
				srcFile.index, path);

			exit(EXIT_FAILURE);
		}

		srcFile.path = path;

		dotPos = dot - path;
		strncpy(srcFile.dstPath, path, dotPos + 1);
		srcFile.dstPath[dotPos + 1] = 'c';
		srcFile.dstPath[dotPos + 2] = '\0';
	}

	//	Try to open the source file.
	//
	fileObj = fopen(path, "r");

	if(fileObj == NULL)
	{
		printf("Error opening the source file (%s): ", path);
		perror(NULL);

		exit(EXIT_FAILURE);
	}

	//	Calculate the size of the source file.
	//
	if(fseek(fileObj, 0, SEEK_END) != 0)
	{
		printf("fseek() error while calculating the size of the file (%s).",
			path);

		exit(EXIT_FAILURE);
	}

	srcFile.size = ftell(fileObj);
	if(srcFile.size == -1L)
	{
		printf("ftell() error while calculating the size of the file (%s): ",
			path);

		perror(NULL);

		exit(EXIT_FAILURE);
	}

	rewind(fileObj);

	//	Allocate memory for the code of source file. It is double sized for
	//	upcoming conversions.
	//
	{
		size_t memSize = sizeof(char) * srcFile.size * 2;

		srcFile.buffer = (char*) malloc (memSize);

		if(srcFile.buffer == NULL)
		{
			printf("Can't allocate dynamic memory for the buffer (%s): %d "
				"bytes.", path, memSize);

			exit(EXIT_FAILURE);
		}

		srcFile.bufSize = memSize;
	}

	//	Copy the content of the source file into the buffer.
	//
	{
		size_t result = fread(srcFile.buffer, sizeof(char), srcFile.size,
			fileObj);

		if(result != srcFile.size)
		{
			printf("fread() error while copying the content of the source "
				"file (%s) into the buffer.", path);

			exit(EXIT_FAILURE);
		}
	}

	//	The source file is not need more.
	//
	fclose(fileObj);
}


//	Write modified file with ".c" extension after code conversions.
//
void writeModFile(void)
{
	//	Pointer to the modified file.
	//
	FILE *fileObj;


	//	Create new file and write modified source code into it.
	//
	fileObj = fopen(srcFile.dstPath, "w");

	if(fileObj == NULL)
	{
		printf("Error opening the destination file (%s): ", srcFile.dstPath);
		perror(NULL);

		exit(EXIT_FAILURE);
	}

	//	Write data.
	//
	{
		size_t charCount;
		size_t result;

		charCount = strlen(srcFile.buffer);
		result = fwrite(srcFile.buffer, sizeof(char), charCount, fileObj);

		if(result != charCount)
		{
			printf("Error. Can't write the buffer to the destination file "
				"(%s).", srcFile.dstPath);

			exit(EXIT_FAILURE);
		}
	}

	//	The destination file is not need more.
	//
	fclose(fileObj);
}

