

#define UNICODE
#define _UNICODE


#include <stdio.h>
#include <windows.h>
#include <commctrl.h>
#include <wincon.h>
#include "window.rc.h"



#define APP_IDENAME "StrictAL IDE"
#define APP_HOMEPAGE "https://www.heroesden.link"

//	Maximum length of the project's name including '\0'.
//
#define IDE_PROJECTNAME_MAXLEN 32

#define APP_FILES_MAX 64
#define APP_FILENAME_MAXLEN 32


//	Maximum length of EDIT controls.
//
#define IDE_CTRL_EDIT_MAXLEN 256



HINSTANCE IDE_hInstance = NULL;
HWND IDE_hConsole = NULL;
HWND IDE_hWndMain = NULL;
HFONT IDE_hfGUI = NULL;


HWND IDE_hDlgFileNewproject = NULL;


HWND IDE_hDlgHelpAbout = NULL;
HWND IDE_hDlgHelpAbout_Build = NULL;
HWND IDE_hDlgHelpAbout_Homepage = NULL;


//	Pointers to original window procedures of controls.
//
WNDPROC IDE_wpBaseButton = NULL;
WNDPROC IDE_wpBaseStatic = NULL;
WNDPROC IDE_wpBaseEdit = NULL;




HWND App_hWndFileList = 0;
HWND App_hWndEditorTabs = 0;

//	Position of an item of menu is counted from zero (0).
//
HMENU IDE_hMainMenu = NULL;
HMENU IDE_hSubmenuProject = NULL;




char App_ProjectName[IDE_PROJECTNAME_MAXLEN] = "";
char App_ProjectFiles[APP_FILES_MAX][APP_FILENAME_MAXLEN] = {0};



//	All virtual keys must be only in upper case.
//
ACCEL App_accelList[] =
{
	{ FCONTROL | FVIRTKEY, 'N', ID_MAINMENU_FILE_NEWPROJECT }, // CTRL + N
	{ FCONTROL | FVIRTKEY, 'A', ID_CMD_SELECT_ALL }, // CTRL + A
	{ FCONTROL | FVIRTKEY, 'C', ID_CMD_COPY }, // CTRL + C
	{ FCONTROL | FVIRTKEY, 'X', ID_CMD_CUT }, // CTRL + X
	{ FCONTROL | FVIRTKEY, 'V', ID_CMD_PASTE } // CTRL + V
};
HACCEL App_hAccel = NULL;





//	Saved state of an EDIT control. It contains control's identifier, inputed
//	text and positions of beginning and ending of selection. Such states are
//	used when an user pressed CTRL-Z combination for undoing last edit.
//
typedef struct _IDE_CTRL_EDIT_STATE
{
	int id;
	char text[IDE_CTRL_EDIT_MAXLEN];
	int selStart;
	int selEnd;
} IDE_CTRL_EDIT_STATE;





//	Non-standard message sent to a parent window by an EDIT control for saving
//	information about current state of the control.
//
//	wParam is NULL.
//	lParam is HWND of the control.
//
#define WM_APP_SAVE_CTRL_EDIT_STATE (WM_APP + 1)

//	Non-standard message sent to a parent window by an EDIT control for loading
//	information about previous state of the control.
//
//	wParam is NULL.
//	lParam is HWND of the control.
//
#define WM_APP_LOAD_CTRL_EDIT_STATE (WM_APP + 2)







void IDE_Error(WCHAR *msg);
void IDE_Warn(char *msg);
void IDE_Log(char *msg);
void IDE_AddConsole(void);
void IDE_RegisterWndClasses(void);
void IDE_CreateWnd_Main(int nCmdShow);
void IDE_CreateDlg_HelpAbout(void);
void IDE_CreateDlg_FileNewproject(void);
void App_CreateAccelTable(void);
void App_CreateFileList(void);
void App_CreateEditorTabs(void);
void App_ClickOnMenu_Help_About(void);
void App_ClickOnMenu_File_Newproject(void);
void App_ClickOn_Project_Newfile(HWND hwnd);
void IDE_CenterWindow(HWND hwnd);
void App_CenterDialogBoxes(WPARAM wParam, LPARAM lParam);
void IDE_SetDefFont(HWND hWnd);
void IDE_ClickOnControl(HWND hParent, int nCtrlId);
void IDE_ClickOnClose(HWND hWnd);
void IDE_SetFocusOnOK(HWND hWnd);
void IDE_RotateFocus(HWND hParent, const int *table, int num);
void IDE_SaveCtrlEditState(HWND hEdit, IDE_CTRL_EDIT_STATE *table, int rows);
void IDE_LoadCtrlEditState(HWND hEdit, IDE_CTRL_EDIT_STATE *table, int rows);
void IDE_PurgeCtrlStateTable(IDE_CTRL_EDIT_STATE *table, int rows);
BOOL IDE_CopyTextToClipboard(char *text);
HWND App_AddTooltip(HWND hParent, UINT childId, char *text);





LRESULT CALLBACK IDE_wpMainWnd(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{

		case WM_ACTIVATE:
		{
			//App_CenterDialogBoxes(wParam, lParam);

			char classname[256];
			HWND hChild = (HWND) lParam;

			//	Is focus switched from main window to child dialog box?
			//
			if(wParam == WA_INACTIVE)
			{

				GetClassName(hChild, classname, sizeof(classname) - 1);
	
				//	Classname for all common dialog boxes is "#32770".
				//
				if(strcmp(classname, "#32770") == 0)
				{
					IDE_CenterWindow(hChild);
					return 0;
				}

				//if(strcmp(classname, "wcModalDialog") == 0)
				//{
				//	App_CenterWindow(hChild);
				//	EnableWindow(hwnd, FALSE);
				//	return 0;
				//}
			}


		}
		break;


		case WM_NOTIFY:
		{
		}
		break;


		//
		//
		case WM_CREATE:
		break;


		//
		//
		case WM_COMMAND:

			switch(LOWORD(wParam))
			{
				case ID_CMD_SELECT_ALL:
				IDE_Log("Testing");
				break;


				case ID_CMD_COPY:
				{
					SendMessage(GetFocus(), WM_COPY, NULL, NULL);
					return 0;
				}
				//break;


				case ID_CMD_CUT:
				{
					SendMessage(GetFocus(), WM_CUT, NULL, NULL);
					return 0;
				}
				//break;


				case ID_CMD_PASTE:
				{
					SendMessage(GetFocus(), WM_PASTE, NULL, NULL);
					return 0;
				}
				//break;


				case ID_MAINMENU_FILE_NEWPROJECT:
				App_ClickOnMenu_File_Newproject();
				break;

				case ID_MAINMENU_FILE_EXIT:
				PostMessage(hwnd, WM_CLOSE, 0, 0);
				break;

				case ID_MAINMENU_PROJECT_NEWFILE:
				App_ClickOn_Project_Newfile(hwnd);
				break;

				case ID_MAINMENU_HELP_ABOUT:
				App_ClickOnMenu_Help_About();
				break;
            }
        break;


		//
		//
		case WM_LBUTTONDOWN:
		break;


		//case WM_SYSCOMMAND:
		//{
			
			//if((wParam & 0xFFF0) == SC_CLOSE)
			//{
			//	EndDialog (hDlg, TRUE);
	         //   return(TRUE);
			//}
        //}
        //break;


		//
		//
		case WM_CLOSE:
		DestroyWindow(hwnd);
		break;


		//
		//
		case WM_DESTROY:
		PostQuitMessage(0);
		break;


		//
		//
        default:
		return DefWindowProc(hwnd, msg, wParam, lParam);
	}

	return 0;
}


//	Base window procedure for all dialogs.
//
LRESULT CALLBACK IDE_wpModalDlg(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
		case WM_SHOWWINDOW:
		{
			//	The dialog window is being shown because ShowWindow() function
			//	was called.
			//
			if(wParam == TRUE && lParam == 0)
			{
				EnableWindow(IDE_hWndMain, FALSE);
				IDE_CenterWindow(hWnd);
			}

			return 0;
		}
		//break;


		case WM_CLOSE:
		{
			//	Hide dialog (not destroy) and return focus to the parent
			//	window.
			//
			HWND hParent = GetParent(hWnd);

			EnableWindow(hParent, TRUE);
			ShowWindow(hWnd, SW_HIDE);

			return 0;
		}
		//break;
	}

	return CallWindowProc(DefWindowProc, hWnd, msg, wParam, lParam);
}


//	Subclass window procedure for <Help - About> dialog derived from
//	IDE_wpModalDlg().
//
LRESULT CALLBACK IDE_wpModalDlg_HelpAbout(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	const int focusTable[] =
	{
		ID_DLG_ABOUT_HELP_URL,
		IDOK
	};


	switch(msg)
	{
		case WM_CHAR:
		{
			if(wParam == VK_TAB)
			{
				IDE_RotateFocus(hWnd, focusTable, (sizeof(focusTable) / sizeof(focusTable[0])));
				return 0;
			}
			if(wParam == VK_RETURN)
			{
				//	Click on OK button.
				//
				IDE_ClickOnControl(hWnd, IDOK);
				return 0;
			}
			if(wParam == VK_ESCAPE)
			{
				SendMessage(hWnd, WM_CLOSE, NULL, NULL);
				return 0;
			}
		}
		break;

		case WM_COMMAND:
		{
			WORD control = LOWORD(wParam);

			//	The user pressed button with homepage URL.
			//
			if(control == ID_DLG_ABOUT_HELP_URL)
			{
				ShellExecute(NULL, "open", APP_HOMEPAGE, NULL, NULL, SW_SHOWNORMAL);
				return 0;
			}

			//	The user pressed OK button.
			//
			if(control == IDOK)
			{
				SendMessage(hWnd, WM_CLOSE, NULL, NULL);
				return 0;
			}
		}
		break;
	}

	return CallWindowProc(IDE_wpModalDlg, hWnd, msg, wParam, lParam);
}





//	Subclass window procedure for <File - New project> dialog.
//
LRESULT CALLBACK IDE_wpModalDlg_FileNewproject(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	const int focusTable[] =
	{
		ID_DLG_FILE_NEWPROJECT_NAME,
		IDOK,
		IDCANCEL
	};


	static IDE_CTRL_EDIT_STATE stateTable[] =
	{
		{ ID_DLG_FILE_NEWPROJECT_NAME, "", 0, 0 }
	};


	switch(msg)
	{
		case WM_CHAR:
		{

			if(wParam == VK_TAB)
			{
				int rows = sizeof(focusTable) / sizeof(focusTable[0]);
				IDE_RotateFocus(hWnd, focusTable, rows);
				return 0;
			}

			if(wParam == VK_RETURN)
			{
				IDE_ClickOnControl(hWnd, IDOK);
				return 0;
			}

			if(wParam == VK_ESCAPE)
			{
				IDE_ClickOnControl(hWnd, IDCANCEL);
				return 0;
			}
		}
		break;


		case WM_COMMAND:
		{
			WORD control = LOWORD(wParam);
			WORD action = HIWORD(wParam);


			if(control == IDCANCEL)
			{
				SendMessage(hWnd, WM_CLOSE, NULL, NULL);
				return 0;
			}

			if(control == ID_DLG_FILE_NEWPROJECT_NAME && action == EN_CHANGE)
			{
				//IDE_Log("CHANGE.");
				return 0;
			}

			if(control == ID_DLG_FILE_NEWPROJECT_NAME && action == EN_UPDATE)
			{
				//IDE_Log("UPDATE.");
				return 0;
			}
		}
		break;


		case WM_CLOSE:
		{
			SetDlgItemText(hWnd, ID_DLG_FILE_NEWPROJECT_NAME, "");
			IDE_PurgeCtrlStateTable(stateTable, sizeof(stateTable) / sizeof(stateTable[0]));

			//	Do not terminate the function, it is needed to proccess
			//	WM_CLOSE by parent WNDPROC.
			//
		}
		break;


		case WM_APP_SAVE_CTRL_EDIT_STATE:
		{
			IDE_SaveCtrlEditState((HWND) lParam, stateTable, sizeof(stateTable) / sizeof(stateTable[0]));
			return 0;
		}
		//break;


		case WM_APP_LOAD_CTRL_EDIT_STATE:
		{
			IDE_LoadCtrlEditState((HWND) lParam, stateTable, sizeof(stateTable) / sizeof(stateTable[0]));
			return 0;
		}
		//break;
	}


	return CallWindowProc(IDE_wpModalDlg, hWnd, msg, wParam, lParam);
}


















LRESULT CALLBACK IDE_wpButton(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{


		case WM_CREATE:
		{
			IDE_SetDefFont(hWnd);
			return 0;
		}
		//break;


		case WM_CHAR:
		{
			//	Forward pressing of ESC and TAB keys to a parent window.
			//
			if(wParam == VK_ESCAPE || wParam == VK_TAB)
			{
				SendMessage(GetParent(hWnd), msg, wParam, lParam);
				return 0;
			}

			//	Emulate click of mouse's left button on the control.
			//
			if(wParam == VK_RETURN)
			{
				IDE_ClickOnControl(hWnd, NULL);
				return 0;
			}
		}
		break;
	}

	return CallWindowProc(IDE_wpBaseButton, hWnd, msg, wParam, lParam);
}



LRESULT CALLBACK IDE_wpStatic(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
		case WM_CREATE:
		{
			IDE_SetDefFont(hWnd);
			return 0;
		}
		//break;
	}

	return CallWindowProc(IDE_wpBaseStatic, hWnd, msg, wParam, lParam);
}



LRESULT CALLBACK IDE_wpEdit(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
		case WM_SHOWWINDOW:
		{
			//	An edit control can't change font on WM_CREATE event, so it is
			//	done here.
			//
			IDE_SetDefFont(hWnd);
			return 0;
		}
		//break;


		case WM_KEYDOWN:
		{
			//	Pressing on DELETE key can be handled only in WM_KEYDOWN event.
			//
			if(wParam == VK_DELETE)
			{
				DWORD selStart, selEnd;
				SendMessage(hWnd, EM_GETSEL, (WPARAM)&selStart, (LPARAM)&selEnd);

				//	If there is no text selection in the EDIT just pass
				//	processing to the base window procedure.
				//
				if(selStart == selEnd) break;

				//	If text has selection then pass processing to WM_CLEAR
				//	handler for saving control state and removing extra spaces.
				//
				SendMessage(hWnd, WM_CLEAR, NULL, NULL);
				return 0;
			}
		}
		break;


		case WM_CHAR:
		{
			DWORD selStart, selEnd;
			char buf[IDE_CTRL_EDIT_MAXLEN];
			DWORD bufLen;
			DWORD i;


			//	Forward some pressing keys to a parent window.
			//
			if(wParam == VK_ESCAPE || wParam == VK_TAB || wParam == VK_RETURN)
			{
				SendMessage(GetParent(hWnd), msg, wParam, lParam);
				return 0;
			}


			//	Logic is the same as for DELETE key.
			//
			if(wParam == VK_BACK)
			{
				DWORD selStart, selEnd;
				SendMessage(hWnd, EM_GETSEL, (WPARAM)&selStart, (LPARAM)&selEnd);

				if(selStart == selEnd) break;

				SendMessage(hWnd, WM_CLEAR, NULL, NULL);
				return 0;
			}


			//	Virtual key #26 is CTRL-Z combination. It is used to save
			//	current state of the EDIT and load previous.
			//
			if(wParam == 26)
			{
				SendMessage(GetParent(hWnd), WM_APP_LOAD_CTRL_EDIT_STATE, wParam, (LPARAM)hWnd);
				return 0;
			}


			if(wParam == VK_SPACE)
			{
				SendMessage(hWnd, EM_GETSEL, (WPARAM)&selStart, (LPARAM)&selEnd);

				//	The user want to add SPACE in the beginning of an EDIT
				//	control. Cancel this operation.
				//
				if(selStart == 0 && selEnd == 0) return 0;

				//	The user want to add SPACE somewhere in the EDIT, but not
				//	in the beginning. Don't get worried about it, it is not
				//	important.
				//
				if(selStart != 0) break;

				//	The user selected text in an EDIT control from the
				//	beginning and want to replace it by SPACE. Here we stretch
				//	the selection behind all following spaces.
				//
				if(selStart == 0 && selEnd != 0)
				{
					GetWindowText(hWnd, buf, sizeof(buf));

					bufLen = strlen(buf);

					for(i = selEnd; (i < bufLen) && (buf[i] == ' '); i++);

					if(i != selEnd)
						SendMessage(hWnd, EM_SETSEL, (WPARAM)selStart, (LPARAM)i);

					wParam = VK_BACK;
					break;
				}
			}

			//	Suppress non-latin characters from input.
			//
			if(wParam < 32 || wParam > 127) return 0;
		}
		break;


		case WM_SYSCHAR:
		{
			//	Key combination ALT-BACKSPACE is UNDO operation (same as
			//	CTRL-Z).
			//
			//if(wParam == VK_BACK)
			//{
			//	SendMessage(GetParent(hWnd), WM_APP_LOAD_CTRL_EDIT_STATE, wParam, (LPARAM)hWnd);
			//	return 0;
			//}
		}
		break;


		//	Remark about WM_CLEAR message. It is sent only by programmer and
		//	only when there is a text selection in the EDIT control.
		//
		case WM_CLEAR:
		{
			DWORD selStart, selEnd;
			char text[IDE_CTRL_EDIT_MAXLEN];
			char buf[IDE_CTRL_EDIT_MAXLEN];
			int i;
			int len;


			SendMessage(hWnd, EM_GETSEL, (WPARAM)&selStart, (LPARAM)&selEnd);

			//	There is no selection in the EDIT.
			//
			if(selStart == selEnd)
			{
				IDE_Warn("An EDIT control without text selection received WM_CLEAR message.");
				return 0;
			}

			//	Send message to a parent window to save current EDIT state.
			//
			SendMessage(GetParent(hWnd), WM_APP_SAVE_CTRL_EDIT_STATE, NULL, (LPARAM)hWnd);

			//	Copy text before user's selection.
			//
			GetWindowText(hWnd, text, sizeof(text));

			strncpy(buf, text, selStart);
			buf[selStart] = '\0';

			//	If selection starting with beginning of the text then need to
			//	calculate number of spaces following right after selection
			//	ending.
			//
			i = 0;

			if(selStart == 0)
			{
				len = strlen(text + selEnd);
				while((i < len) && (text[selEnd + i] == ' ')) i++;
			}

			strcat(buf, text + selEnd + i);

			//	Change text in the EDIT and move caret.
			//
			SetWindowText(hWnd, buf);
			SendMessage(hWnd, EM_SETSEL, selStart, selStart);

			return 0;
		}
		break;


		case WM_PASTE:
		{
			return 0;
		}
		//break;


		case WM_CUT:
		{
			DWORD selStart, selEnd;
			char text[IDE_CTRL_EDIT_MAXLEN];
			char buf[IDE_CTRL_EDIT_MAXLEN];
			int i;
			int len;


			SendMessage(hWnd, EM_GETSEL, (WPARAM)&selStart, (LPARAM)&selEnd);

			//	There is no selection in the EDIT.
			//
			if(selStart == selEnd) return 0;

			GetWindowText(hWnd, text, sizeof(text));

			//	Copy text to a clipboard.
			//
			len = selEnd - selStart;
			strncpy(buf, text + selStart, len);
			buf[len] = '\0';

			if(IDE_CopyTextToClipboard(buf) == FALSE) return 0;

			//	Send message to a parent window to save current EDIT state.
			//
			SendMessage(GetParent(hWnd), WM_APP_SAVE_CTRL_EDIT_STATE, NULL, (LPARAM)hWnd);

			//	Copy text before user's selection.
			//
			strncpy(buf, text, selStart);
			buf[selStart] = '\0';

			//	If selection starting with beginning of the text then need to
			//	calculate number of spaces following right after selection
			//	ending.
			//
			i = 0;

			if(selStart == 0)
			{
				len = strlen(text + selEnd);
				while((i < len) && (text[selEnd + i] == ' ')) i++;
			}

			strcat(buf, text + selEnd + i);

			//	Change text in the EDIT and move caret.
			//
			SetWindowText(hWnd, buf);
			SendMessage(hWnd, EM_SETSEL, selStart, selStart);

			return 0;
		}
		//break;
	}

	return CallWindowProc(IDE_wpBaseEdit, hWnd, msg, wParam, lParam);
}









int WINAPI wWinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow)
{
	MSG Msg;
	IDE_hInstance = hInst;


	IDE_AddConsole();


	IDE_hfGUI = GetStockObject(DEFAULT_GUI_FONT);
	if(IDE_hfGUI == NULL)
		IDE_Error(L"Can't get DEFAULT_GUI_FONT stock object.");


	InitCommonControls();


	IDE_RegisterWndClasses();
	IDE_CreateWnd_Main(nCmdShow);
	IDE_CreateDlg_HelpAbout();
	IDE_CreateDlg_FileNewproject();

	App_CreateAccelTable();
	App_CreateFileList();
	App_CreateEditorTabs();


	while(GetMessage(&Msg, NULL, 0, 0) > 0)
	{
		if(TranslateAccelerator(IDE_hWndMain, App_hAccel, &Msg)) continue;

		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}


	(void) hPrevInst;
	(void) lpCmdLine;
	return Msg.wParam;
}




void IDE_Error(WCHAR *msg)
{
	WCHAR buf[256];
	const size_t count = sizeof(buf) / sizeof(buf[0]) - 1;

	snwprintf(buf, count, L"%s\n\nError code: %lu.", msg, GetLastError());
	MessageBoxW(IDE_hWndMain, buf, L"Critical error", MB_OK | MB_ICONERROR);

	exit(EXIT_FAILURE);
}




void IDE_Warn(char *msg)
{
	char buf[256];

	snprintf(buf, sizeof(buf), "%s\n\nError code: %lu.", msg, GetLastError());
	MessageBox(IDE_hWndMain, buf, "Non-critical error", MB_OK | MB_ICONWARNING);
}





void IDE_Log(char *msg)
{
	HANDLE hOut;
	char buf[256];
	static int i = 0;

	if(IDE_hConsole == NULL)
		IDE_Error(L"Can't write log (the console is not allocated).");

	snprintf(buf, sizeof(buf), "#%d %s\n", i, msg);

	hOut = GetStdHandle(STD_OUTPUT_HANDLE);
	WriteConsole(hOut, buf, strlen(buf), NULL, NULL);

	i++;
}






void IDE_AddConsole(void)
{
	HMONITOR hMon;
	MONITORINFO monInfo;

	HANDLE hOut;
	CONSOLE_SCREEN_BUFFER_INFO conInfo;
	COORD bufSize;
	RECT rcCon;
	int width, height, x, y;
	UINT flags;







	{
		WCHAR argv[256] = { L'T', L'e', L's', L't', L'!', L'\0' };
		WCHAR buf[256];
		STARTUPINFOW si = {0};
		PROCESS_INFORMATION pi = {0};

		//info.cb = sizeof(info);
		//GetStartupInfoW(&info);

		//swprintf(buf, L"title = %s, dwXCountChars = %d", info.lpTitle, info.dwXCountChars);
		//IDE_Error(buf);


		si.cb = sizeof(si);

		if(!CreateProcessW(NULL, L"cmd.exe", NULL, NULL, TRUE, CREATE_NEW_CONSOLE, NULL, NULL, &si, &pi))
			IDE_Error(L"Can't create a process for a console.");



		return;
	}








	//	Allocate a console.
	//
	if(AllocConsole() == 0)
		IDE_Error(L"Can't allocate console.");


	//	Make handle of the console global.
	//
	IDE_hConsole = FindWindowW(L"ConsoleWindowClass", NULL);
	if(IDE_hConsole == NULL)
		IDE_Error(L"Can't find handle of the console.");


	//	Change buffer size of the console.
	//
	hOut = GetStdHandle(STD_OUTPUT_HANDLE);

	if(hOut == NULL || hOut == INVALID_HANDLE_VALUE)
		IDE_Error(L"Can't get STD_OUTPUT_HANDLE.");


	if(GetConsoleScreenBufferInfo(hOut, &conInfo) == 0)
		IDE_Error(L"Can't get information about console buffer.");

	//	Don't change X here because it is pointless. The operating system
	//	doesn't get you to do it. You need to change size of the window not a
	//	buffer.
	//
	//	I wish it have to be like this, but no luck:
	//	bufSize.X = (conInfo.dwSize.X < 80) ? 80 : conInfo.dwSize.X;
	//
	bufSize.X = conInfo.dwSize.X;
	bufSize.Y = (conInfo.dwSize.Y > 240) ? 240 : conInfo.dwSize.Y;

	if(SetConsoleScreenBufferSize(hOut, bufSize) == 0)
		IDE_Error(L"Can't change buffer size of the console.");


	//	Determine active monitor and move the console.
	//
	hMon = MonitorFromWindow(IDE_hConsole, MONITOR_DEFAULTTOPRIMARY);

	monInfo.cbSize = sizeof(monInfo);
	if(GetMonitorInfoW(hMon, &monInfo) == 0)
		IDE_Error(L"Can't get information about monitor while attaching the console.");

	x = monInfo.rcWork.left;
	y = monInfo.rcWork.top;

	if(GetWindowRect(IDE_hConsole, &rcCon) == 0)
		IDE_Error(L"Can't get window size of the console.");


	//	Try to normalize width of the console to about 80 characters (79
	//	characters of a text + '\n').
	//
	if(bufSize.X == 80)
	{
		width = rcCon.right - rcCon.left;
	}
	else
	{
		width = (rcCon.right - rcCon.left) * (80.0 / (bufSize.X - 3));
	}

	height = monInfo.rcWork.bottom - monInfo.rcWork.top;

	flags = (IDE_hWndMain == NULL) ? SWP_NOZORDER : NULL;

	if(SetWindowPos(IDE_hConsole, IDE_hWndMain, x, y, width, height, flags) == 0)
		IDE_Error(L"Can't move the console.");
}







void IDE_RegisterWndClasses(void)
{
	HCURSOR cursor = LoadCursor(NULL, IDC_ARROW);


	//	Window class for main window.
	//
	{
		WNDCLASSEX wc = {0};

		wc.cbSize = sizeof(wc);
		wc.style = NULL;
		wc.lpfnWndProc = IDE_wpMainWnd;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = IDE_hInstance;
		wc.hIcon = LoadIcon(IDE_hInstance, MAKEINTRESOURCE(ID_APP_ICON));
		wc.hCursor = cursor;
		wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
		wc.lpszMenuName = NULL;
		wc.lpszClassName = L"wcMain";
		wc.hIconSm = (HICON)LoadImage(IDE_hInstance, MAKEINTRESOURCE(ID_APP_ICON), IMAGE_ICON, 16, 16, NULL);

		if(!RegisterClassEx(&wc))
			IDE_Error(L"Can't register 'wcMain' window class.");
	}


	//	Window class for modal dialogs.
	//
	{
		WNDCLASSEX wc = {0};

		wc.cbSize = sizeof(wc);
		wc.style = NULL;
		wc.lpfnWndProc = IDE_wpModalDlg;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = IDE_hInstance;
		wc.hIcon = NULL;
		wc.hCursor = cursor;
		wc.hbrBackground = (HBRUSH)(COLOR_MENU + 1);
		wc.lpszMenuName = NULL;
		wc.lpszClassName = L"wcModalDialog";
		wc.hIconSm = NULL;

		if(!RegisterClassEx(&wc))
			IDE_Error(L"Can't register 'wcModalDialog' window class.");
	}


	//	Window superclass for BUTTONs.
	//
	{
		WNDCLASSEX wc = {0};

		wc.cbSize = sizeof(wc);
		if(GetClassInfoEx(NULL, L"BUTTON", &wc) == FALSE)
			IDE_Error(L"Can't get information about <BUTTON> class.");

		IDE_wpBaseButton = wc.lpfnWndProc;

		wc.lpfnWndProc = IDE_wpButton;
		wc.hInstance = IDE_hInstance;
		wc.lpszClassName = L"wcButton";

		if(!RegisterClassEx(&wc))
			IDE_Error(L"Can't register 'wcButton' window class.");
	}


	//	Window superclass for STATIC text.
	//
	{
		WNDCLASSEX wc = {0};

		wc.cbSize = sizeof(wc);
		if(GetClassInfoEx(NULL, L"STATIC", &wc) == FALSE)
			IDE_Error(L"Can't get information about <STATIC> class.");

		IDE_wpBaseStatic = wc.lpfnWndProc;

		wc.lpfnWndProc = IDE_wpStatic;
		wc.hInstance = IDE_hInstance;
		wc.lpszClassName = L"wcStatic";

		if(!RegisterClassEx(&wc))
			IDE_Error(L"Can't register 'wcStatic' window class.");
	}


	//	Window superclass for EDIT input.
	//
	{
		WNDCLASSEX wc = {0};

		wc.cbSize = sizeof(wc);
		if(GetClassInfoEx(NULL, L"EDIT", &wc) == FALSE)
			IDE_Error(L"Can't get information about <EDIT> class.");
	
		IDE_wpBaseEdit = wc.lpfnWndProc;

		wc.lpfnWndProc = IDE_wpEdit;
		wc.hInstance = IDE_hInstance;
		wc.lpszClassName = L"wcEdit";

		if(!RegisterClassEx(&wc))
			IDE_Error(L"Can't register 'wcEdit' window class.");
	}
}










void IDE_CreateWnd_Main(int nCmdShow)
{

	IDE_hWndMain = CreateWindowEx
	(
		WS_EX_CLIENTEDGE,
		L"wcMain",
		L"StrictAL IDE",
		WS_OVERLAPPEDWINDOW,
		CW_USEDEFAULT, CW_USEDEFAULT, 400, 300,
		NULL, NULL, IDE_hInstance, NULL
	);


	if(IDE_hWndMain == NULL)
		IDE_Error(L"Can't create main window.");


	//	Add menu to the main window.
	//
	{
		IDE_hMainMenu = CreateMenu();

		if(IDE_hMainMenu == NULL)
			IDE_Error(L"Can't create menu for main window.");

		//	Menu -> File -> ...
		//
		{
			HMENU hSubmenu;
			hSubmenu = CreatePopupMenu();

			if(hSubmenu == NULL)
				IDE_Error(L"Can't create <File> popup menu for main window.");

			if(!AppendMenuW(IDE_hMainMenu, MF_STRING | MF_POPUP, (UINT)hSubmenu, L"&File"))
				IDE_Error(L"Can't create <File> menu item for main window.");

			if(!AppendMenuW(hSubmenu, MF_STRING, ID_MAINMENU_FILE_NEWPROJECT, L"&New project\tCtrl+N"))
				IDE_Error(L"Can't create <File-New project> menu item for main window.");

			if(!AppendMenuW(hSubmenu, MF_SEPARATOR, -1, L""))
				IDE_Error(L"Can't create separator in <File> menu of main window.");

			if(!AppendMenuW(hSubmenu, MF_STRING, ID_MAINMENU_FILE_EXIT, L"E&xit"))
				IDE_Error(L"Can't create <File-Exit> menu item for main window.");
		}

		//	Menu -> Project -> ...
		//
		{
			HMENU hSubmenu;
			hSubmenu = CreatePopupMenu();

			if(hSubmenu == NULL)
				IDE_Error(L"Can't create <Project> popup menu for main window.");

			if(!AppendMenuW(IDE_hMainMenu, MF_STRING | MF_POPUP | MF_GRAYED, (UINT)hSubmenu, L"Project"))
				IDE_Error(L"Can't create <Project> menu item for main window.");

			if(!AppendMenuW(hSubmenu, MF_STRING, ID_MAINMENU_PROJECT_NEWFILE, L"New file"))
				IDE_Error(L"Can't create <Project-New file> menu item for main window.");

			IDE_hSubmenuProject = hSubmenu;
		}
	
		//	Menu -> Help -> ...
		//
		{
			HMENU hSubmenu;
			hSubmenu = CreatePopupMenu();

			if(hSubmenu == NULL)
				IDE_Error(L"Can't create <Help> popup menu for main window.");

			if(!AppendMenuW(IDE_hMainMenu, MF_STRING | MF_POPUP, (UINT)hSubmenu, L"&Help"))
				IDE_Error(L"Can't create <Help> menu item for main window.");

			if(!AppendMenuW(hSubmenu, MF_STRING, ID_MAINMENU_HELP_ABOUT, L"About"))
				IDE_Error(L"Can't create <Help-About> menu item for main window.");
		}


		SetMenu(IDE_hWndMain, IDE_hMainMenu);
	}


	//	Center main window and show it.
	//
	IDE_CenterWindow(IDE_hWndMain);
	ShowWindow(IDE_hWndMain, nCmdShow);
	UpdateWindow(IDE_hWndMain);
}




//	Create <Help - About> dialog.
//
void IDE_CreateDlg_HelpAbout(void)
{

	IDE_hDlgHelpAbout = CreateWindowEx
	(
		WS_EX_DLGMODALFRAME,
		L"wcModalDialog",
		L"About IDE",
		WS_CAPTION | WS_SYSMENU | WS_POPUP,
		CW_USEDEFAULT, CW_USEDEFAULT, 246, 176,
		IDE_hWndMain, NULL, IDE_hInstance, NULL
	);


	if(IDE_hDlgHelpAbout == NULL)
		IDE_Error(L"Can't create <Help-About> dialog.");


	if(SetWindowLongPtr(IDE_hDlgHelpAbout, GWLP_WNDPROC, (LONG_PTR)IDE_wpModalDlg_HelpAbout) == 0)
		IDE_Error(L"Can't change window procedure of <Help-About> dialog.");


	//	Build date.
	//
	{
		HWND ctrl;
		WCHAR buf[128];
		const size_t count = sizeof(buf) / sizeof(buf[0]) - 1;

		snwprintf(buf, count, "Build: %s %s", __TIME__, __DATE__);

		ctrl = CreateWindowEx(NULL, L"wcStatic", buf, WS_VISIBLE | WS_CHILD | SS_SIMPLE, 15, 18, 150, 20,
			IDE_hDlgHelpAbout, NULL, IDE_hInstance, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <Help-About-Build> static window.");
	}


	//	Homepage.
	//
	{
		HWND ctrl;

		ctrl = CreateWindowEx(NULL, L"wcStatic", L"Homepage:", WS_VISIBLE | WS_CHILD | SS_SIMPLE, 15, 49, 150, 20,
			IDE_hDlgHelpAbout, NULL, IDE_hInstance, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <Help-About-Homepage> static window.");
	}


	//	Homepage URL.
	//
	{
		HWND ctrl;

		ctrl = CreateWindowEx(NULL, L"wcButton", APP_HOMEPAGE, WS_VISIBLE | WS_CHILD | BS_FLAT, 75, 44, 150, 24,
			IDE_hDlgHelpAbout, (HMENU)ID_DLG_ABOUT_HELP_URL, IDE_hInstance, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <Help-About-URL> button.");
	}


	//	OK button.
	//
	{
		HWND ctrl;

		ctrl = CreateWindowEx(NULL, L"wcButton", L"OK", WS_VISIBLE | WS_CHILD, 155, 108, 70, 24,
			IDE_hDlgHelpAbout, (HMENU)IDOK, IDE_hInstance, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <Help-About-OK> button.");
	}
}










//	Create <File - New project> dialog.
//
void IDE_CreateDlg_FileNewproject(void)
{

	IDE_hDlgFileNewproject = CreateWindowEx
	(
		WS_EX_DLGMODALFRAME,
		L"wcModalDialog",
		L"Create new project",
		WS_CAPTION | WS_SYSMENU | WS_POPUP,
		CW_USEDEFAULT, CW_USEDEFAULT, 246, 164,
		IDE_hWndMain, NULL, IDE_hInstance, NULL
	);


	if(IDE_hDlgFileNewproject == NULL)
		IDE_Error(L"Can't create <File-New project> dialog.");


	if(SetWindowLongPtr(IDE_hDlgFileNewproject, GWLP_WNDPROC, (LONG_PTR)IDE_wpModalDlg_FileNewproject) == 0)
		IDE_Error(L"Can't change window procedure of <File-New project> dialog.");


	//	Project name text.
	//
	{
		HWND ctrl;

		ctrl = CreateWindowEx(NULL, L"wcStatic", L"Project's name:", WS_VISIBLE | WS_CHILD | SS_SIMPLE, 15, 18, 150, 20,
			IDE_hDlgFileNewproject, NULL, IDE_hInstance, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <File-New project-Name> static.");
	}


	//	Project name input.
	//
	{
		HWND ctrl;

		ctrl = CreateWindowEx(WS_EX_CLIENTEDGE, L"wcEdit", "", WS_VISIBLE | WS_CHILD | ES_LEFT | ES_AUTOHSCROLL, 15, 36, 210, 20,
			IDE_hDlgFileNewproject, (HMENU)ID_DLG_FILE_NEWPROJECT_NAME, IDE_hInstance, NULL);

		SendMessage(ctrl, EM_LIMITTEXT, IDE_PROJECTNAME_MAXLEN - 1, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <File-New project-Name> edit.");
	}


	//	OK button.
	//
	{
		HWND ctrl;

		ctrl = CreateWindowEx(NULL, L"wcButton", L"OK", WS_VISIBLE | WS_CHILD | WS_DISABLED, 78, 96, 70, 24,
			IDE_hDlgFileNewproject, (HMENU)IDOK, IDE_hInstance, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <File-New project-OK> button.");
	}


	//	Cancel button.
	//
	{
		HWND ctrl;

		ctrl = CreateWindowEx(NULL, L"wcButton", L"Cancel", WS_VISIBLE | WS_CHILD, 155, 96, 70, 24,
			IDE_hDlgFileNewproject, (HMENU)IDCANCEL, IDE_hInstance, NULL);

		if(ctrl == NULL)
			IDE_Error(L"Can't create <File-New project-Cancel> button.");
	}
}
























void App_CreateAccelTable(void)
{
	App_hAccel = CreateAcceleratorTable(App_accelList, sizeof(App_accelList) / sizeof(ACCEL));

	if(App_hAccel == NULL)
		IDE_Error(L"Can't create accelerator table.");
}





//	INFO:
//
//	InitTreeViewImageLists() and InitTreeViewItems() not implement in BCC551.
//
void App_CreateFileList(void)
{
	RECT rcClient;

	GetClientRect(IDE_hWndMain, &rcClient);

	App_hWndFileList = CreateWindowEx
	(
		0,
		WC_TREEVIEW,
		"Tree View",
		WS_VISIBLE | WS_CHILD | WS_BORDER | TVS_HASLINES,
		0,
		0,
		rcClient.right * 0.35,
		rcClient.bottom,
		IDE_hWndMain,
		(HMENU)ID_FILELIST,
		IDE_hInstance,
		NULL
	);

	if(App_hWndFileList == NULL)
		IDE_Error(L"Can't create WC_TREEVIEW control.");
}




void App_CreateEditorTabs(void)
{
	RECT rcClient;
	TCITEM tci = {0};


	GetClientRect(IDE_hWndMain, &rcClient);


	App_hWndEditorTabs = CreateWindowEx
	(
		0,
		WC_TABCONTROL,
		"Editor Tabs",
		WS_CHILD | WS_VISIBLE | WS_BORDER,
		300,
		0,
		rcClient.right,
		rcClient.bottom, 
        IDE_hWndMain,
		(HMENU)ID_EDITORTABS,
		IDE_hInstance,
		NULL
	);


	if (App_hWndEditorTabs == NULL)
		IDE_Error(L"Can't create WC_TABCONTROL control.");


	tci.mask = TCIF_TEXT;
	tci.pszText = "tab1";
	tci.cchTextMax = 4;

	if(TabCtrl_InsertItem(App_hWndEditorTabs, 0, &tci) == -1)
		IDE_Error(L"Can't add item #0 to WC_TABCONTROL.");
}








void App_ClickOnMenu_File_Newproject(void)
{
	HWND hEdit;

	//	Is some project already opened?
	//
	if(strlen(App_ProjectName) != 0)
	{
		MessageBox(IDE_hWndMain, "Some project is already opened. You have to close it before create a new one.", "Error", MB_OK | MB_ICONEXCLAMATION);
		return;
	}

	hEdit = GetDlgItem(IDE_hDlgFileNewproject, ID_DLG_FILE_NEWPROJECT_NAME);

	ShowWindow(IDE_hDlgFileNewproject, SW_NORMAL);
	SetFocus(hEdit);
}





void App_ClickOnMenu_Help_About(void)
{
	ShowWindow(IDE_hDlgHelpAbout, SW_NORMAL);
}











BOOL CALLBACK procClickOn_File_Newproject(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	WORD control = LOWORD(wParam);
	WORD action = HIWORD(wParam);


	//	Initial configuration.
	//
	if(msg == WM_INITDIALOG)
	{
		//	Set maximum number of characters in edit control and set focus on
		//	it.
		//
		HWND hEdit = GetDlgItem(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME);

		SendMessage(hEdit, EM_LIMITTEXT, IDE_PROJECTNAME_MAXLEN - 1, NULL);
		SetFocus(hEdit);

		//	If return TRUE, then first control in the dialogbox will be
		//	focused.
		//
		return FALSE;
	}

	//	Enable CTRL-A combination.
	//
	if(msg == WM_COMMAND && control == ID_CMD_SELECT_ALL)
    {
        //on which control there was pressed Ctrl+A
        //there is no way of getting HWND through wParam and lParam
        //so we get HWND which currently has focus.

        //HWND hFocused = GetFocus();
        //wchar_t className[6];
        //GetClassName(hFocused, className, 6);
        //if (hFocudsed && !wcsicmp(className, L"edit"))
          //  SendMessage(hFocused, EM_SETSEL, 0, -1);

		//App_Log("Hellooooo");
		//return FALSE;
    }


	//	The user change content in edit control.
	//
	if(msg == WM_COMMAND && control == ID_DLG_FILE_NEWPROJECT_NAME && action == EN_CHANGE)
	{
		int nameLen;
		int bufLen;
		int i;
		int begin;
		HWND hOK = GetDlgItem(hwndDlg, IDOK);
		char projectName[IDE_PROJECTNAME_MAXLEN] = "";
		char buf[IDE_PROJECTNAME_MAXLEN] = "";

		GetDlgItemText(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME, projectName, sizeof(projectName) - 1);

		//	Is filename empty?
		//
		nameLen = strlen(projectName);

		if(nameLen == 0)
		{
			EnableWindow(hOK, FALSE);
			return FALSE;
		}

		//	Remove invalid characters from user's input.
		//
		for(i = 0; i < nameLen; i++)
		{
			if(projectName[i] >= 32 && projectName[i] <= 126) strncat(buf, projectName + i, 1);
		}

		bufLen = strlen(buf);

		//	The input contains only invalid characters.
		//
		if(bufLen == 0)
		{
			SetDlgItemText(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME, "");
			EnableWindow(hOK, FALSE);
			return FALSE;
		}

		//	Find first visible character (non space) in the input.
		//
		begin = -1;
		for(i = 0; i < bufLen; i++)
		{
			if(buf[i] != ' ')
			{
				begin = i;
				break;
			}
		}

		//	The input consists of spaces only.
		//
		if(begin == -1)
		{
			SetDlgItemText(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME, "");
			EnableWindow(hOK, FALSE);
			return FALSE;
		}

		//	If the input doesn't contain invalid characters and spaces in the
		//	beginning then all is fine.
		//
		if(begin == 0 && nameLen == bufLen)
		{
			EnableWindow(hOK, TRUE);
			return FALSE;
		}

		//	Remove spaces from the beginning of the input.
		//
		i = bufLen - begin;
		strncpy(projectName, buf + begin, i);
		projectName[i] = '\0';
		SetDlgItemText(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME, projectName);

		//	Move caret to the end of the edit.
		//
		SendDlgItemMessage(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME, EM_SETSEL, i, i);


		EnableWindow(hOK, TRUE);


		return FALSE;
	}


	//	The user pressed "OK" or "Cancel" button.
	//
	if(msg == WM_COMMAND)
	{
		if(control == IDOK)
		{
			char buf[128];
			HWND hSubmenu;



		//	Find the last visible characters (non space) in the input.
		//
		/*
		end = -1;
		for(i = len - 1; i >= 0; i--)
		{
			if(buf[i] != ' ')
			{
				end = i;
				break;
			}
		}*/


			//	Change title of the main window.
			//
			GetDlgItemText(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME, App_ProjectName, sizeof(App_ProjectName) - 1);

			snprintf(buf, sizeof(buf), "%s - %s", APP_IDENAME, App_ProjectName);
			SetWindowText(IDE_hWndMain, buf);

			//	Enable "Project" popup in the main menu.
			//
			EnableMenuItem(IDE_hMainMenu, (UINT)IDE_hSubmenuProject, MF_BYCOMMAND | MF_ENABLED);


			EndDialog(hwndDlg, IDOK);
			return TRUE;
		}
		else if(control == IDCANCEL)
		{
			EndDialog(hwndDlg, IDCANCEL);
			return TRUE;
		}
	}

	(void) lParam;
	return FALSE;
}








BOOL CALLBACK procClickOn_Project_Newfile(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	//	Initial configuration.
	//
	if(msg == WM_INITDIALOG)
	{
		int i;
		HWND hName = GetDlgItem(hwndDlg, ID_DLG_PROJECT_NEWFILE_NAME);
		HWND hCombobox = GetDlgItem(hwndDlg, ID_DLG_PROJECT_NEWFILE_EXT);

		char *extension[] =
		{
			"Source code (*.sals)"
		};

		//SendMessage(hEdit, EM_LIMITTEXT, sizeof(projectName) - 1, 0);
		//SetFocus(hEdit);

		//return FALSE;



		//	Fill up combobox with extensions and select first option.
		//
		for(i = 0; i < (sizeof(extension) / sizeof(char *)); i++)
		{
			SendMessage(hCombobox, CB_ADDSTRING, NULL, (LPARAM)extension[i]);
		}
		SendMessage(hCombobox, CB_SETCURSEL, 0, NULL);


		//	Constrain maximum number of input characters in filename and set
		//	focus.
		//
		SendMessage(hName, EM_LIMITTEXT, APP_FILENAME_MAXLEN - 1, NULL);
		SetFocus(hName);


		return FALSE;
	}

	return FALSE;
}
void App_ClickOn_Project_Newfile(HWND hwnd)
{
	DialogBox(IDE_hInstance, MAKEINTRESOURCE(ID_DLG_PROJECT_NEWFILE), hwnd, (DLGPROC)procClickOn_Project_Newfile);
}














//	Place a window in the center of its parent window. If the window doesn't
//	have a parent, it will be centered by a screen.
//
void IDE_CenterWindow(HWND hWnd)
{
	HWND hParent;
	RECT rcWnd;
	LONG wndWidth, wndHeight;
	int x, y;


	//	Get size of the window.
	//
	GetWindowRect(hWnd, &rcWnd);
	wndWidth = rcWnd.right - rcWnd.left;
	wndHeight = rcWnd.bottom - rcWnd.top;


	hParent = GetParent(hWnd);


	//	Window doesn't have a parent. It will be placed on the center of a
	//	screen.
	//
	if(hParent == NULL)
	{
		HMONITOR hMon;
		MONITORINFO monInfo;
		RECT rcWnd;
		LONG monWidth, monHeight;


		hMon = MonitorFromWindow(hWnd, MONITOR_DEFAULTTOPRIMARY);


		monInfo.cbSize = sizeof(monInfo);
		if(GetMonitorInfoW(hMon, &monInfo) == 0)
			IDE_Error(L"Can't get information about monitor while centering a window.");


		monWidth = monInfo.rcWork.right - monInfo.rcWork.left;
		monHeight = monInfo.rcWork.bottom - monInfo.rcWork.top;


		x = monInfo.rcWork.left + (monWidth - wndWidth) / 2;
		y = monInfo.rcWork.top + (monHeight - wndHeight) / 2;
	}
	//
	//	Window has a parent. It will be placed in the center of parent window.
	//
	else
	{
		RECT rcParent;
		LONG parWidth, parHeight;


		GetWindowRect(hParent, &rcParent);
		parWidth = rcParent.right - rcParent.left;
		parHeight = rcParent.bottom - rcParent.top;


		x = rcParent.left + (parWidth - wndWidth) / 2;
		y = rcParent.top + (parHeight - wndHeight) / 2;
	}


	//	Move the window.
	//
	SetWindowPos(hWnd, 0, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
}






//	Move all uncontrolled dialog boxes to the center of the parent window.
//
void App_CenterDialogBoxes(WPARAM wParam, LPARAM lParam)
{
	char classname[256];
	HWND hChild = (HWND) lParam;

	//	Is focus switched from main window to child dialog box?
	//
	if(wParam != WA_INACTIVE) return;

	GetClassName(hChild, classname, sizeof(classname) - 1);

	//	Classname for all common dialog boxes is "#32770".
	//
	if(strcmp(classname, "#32770") != 0) return;

	IDE_CenterWindow(hChild);
}








void IDE_SetDefFont(HWND hWnd)
{
	SendMessage(hWnd, WM_SETFONT, (WPARAM)IDE_hfGUI, MAKELPARAM(TRUE, 0));
}








//	Emulate clicking of mouse's left button on the center of a control. If
//	<nCtrlId> is NULL then it is assumed that <hParent> is a handler to the
//	target control.
//
void IDE_ClickOnControl(HWND hParent, int nCtrlId)
{
	HWND hControl;
	RECT rcControl;
	int x, y; 

	hControl = (nCtrlId == NULL) ? hParent : GetDlgItem(hParent, nCtrlId);


	//	Is the control enabled?
	//
	if(IsWindowEnabled(hControl) == 0) return;


	GetClientRect(hControl, &rcControl);

	x = rcControl.right / 2;
	y = rcControl.bottom / 2;

	SendMessage(hControl, WM_LBUTTONDOWN, NULL, MAKELPARAM(x, y));
	Sleep(50);
	SendMessage(hControl, WM_LBUTTONUP, NULL, MAKELPARAM(x, y));
}



//	Emulate clicking of mouse's left button on the close button (X) placed in
//	the system menu.
//
void IDE_ClickOnClose(HWND hWnd)
{
	/*
	RECT rcClose = {0};
	HMENU hSysMenu;
	int lastItemId;
	int lastItemPos;
	int i;

	hSysMenu = GetSystemMenu(hWnd, FALSE);
	lastItemPos = GetMenuItemCount(hSysMenu) - 1;
	lastItemId = GetMenuItemID(hSysMenu, lastItemPos);


	if(lastItemId != SC_CLOSE)
	{
		SendMessage(hWnd, WM_CLOSE, NULL, NULL);
		return;
	}

	GetMenuItemRect(hWnd, hSysMenu, lastItemPos, &rcClose);
	*/


	RECT rcWindow;
	int x, y;
	float ratio;

	INPUT inputs[2] = {0};


	GetWindowRect(hWnd, &rcWindow);
	x = rcWindow.right - GetSystemMetrics(SM_CXBORDER) - (GetSystemMetrics(SM_CXSIZE) / 2);
	y = rcWindow.top + GetSystemMetrics(SM_CYBORDER) + (GetSystemMetrics(SM_CYSIZE) / 2);



	/*
    int vx = GetSystemMetrics(SM_XVIRTUALSCREEN);
    int vy = GetSystemMetrics(SM_YVIRTUALSCREEN);
    int vw = GetSystemMetrics(SM_CXVIRTUALSCREEN);
    int vh = GetSystemMetrics(SM_CYVIRTUALSCREEN);

    LONG nx = MulDiv(px - vx, 65535, vw - 1);
    LONG ny = MulDiv(py - vy, 65535, vh - 1);

    INPUT in[3] = {};
    in[0].type = INPUT_MOUSE;
    in[0].mi.dx = nx; in[0].mi.dy = ny;
    in[0].mi.dwFlags = MOUSEEVENTF_MOVE | MOUSEEVENTF_ABSOLUTE | MOUSEEVENTF_VIRTUALDESK;
    in[1].type = INPUT_MOUSE; in[1].mi.dwFlags = MOUSEEVENTF_LEFTDOWN;
    in[2].type = INPUT_MOUSE; in[2].mi.dwFlags = MOUSEEVENTF_LEFTUP;

    SendInput(3, in, sizeof(INPUT));
	*/


	//ratio = x / (float)GetSystemMetrics();

	//	Left button is down.
	//
	inputs[0].type = INPUT_MOUSE;
	inputs[0].mi.dx = x;
	inputs[0].mi.dy = y;
	inputs[0].mi.mouseData = NULL;
	inputs[0].mi.dwFlags = MOUSEEVENTF_ABSOLUTE | MOUSEEVENTF_VIRTUALDESK | MOUSEEVENTF_LEFTDOWN;
	inputs[0].mi.time = 2050;
	inputs[0].mi.dwExtraInfo = NULL;

	//	Left button is up.
	//
	memcpy(&inputs[1], &inputs[0], sizeof(inputs[1]));
	inputs[1].mi.dwFlags = MOUSEEVENTF_ABSOLUTE | MOUSEEVENTF_VIRTUALDESK | MOUSEEVENTF_LEFTUP;
	inputs[1].mi.time = 2000;


	SendInput(sizeof(inputs) / sizeof(inputs[0]), inputs, sizeof(inputs[0]));


	{
		char buf[128];
		sprintf(buf, "x = %u, y = %u, w = %u, h = %u", GetSystemMetrics(SM_XVIRTUALSCREEN), GetSystemMetrics(SM_YVIRTUALSCREEN),
					GetSystemMetrics(SM_CXVIRTUALSCREEN), GetSystemMetrics(SM_CYVIRTUALSCREEN));
		IDE_Log(buf);
	}
}







//	Move focus to OK button.
//
void IDE_SetFocusOnOK(HWND hWnd)
{
	HWND hOK = GetDlgItem(hWnd, IDOK);
	SetFocus(hOK);
}









//	Rotate focus among controls of parent window. This function is called when
//	an user presses TAB key.
//
//	hParent - HWND of parent window.
//	table - pointer to rotation table defined in parent WNDPROC.
//	num - number of rows in rotation table.
//
void IDE_RotateFocus(HWND hParent, const int *table, int num)
{
	int i;
	HWND hFocusedCtrl;
	HWND hNextCtrl;
	int nCtrlId;
	int nCtrlPos = -1;

	hFocusedCtrl = GetFocus();

	//	Focus is on a dialog window, not a control.
	//
	if(hParent == hFocusedCtrl)
	{
		//	Move focus on the first non-disabled control from the table.
		//
		for(i = 0; i < num; i++)
		{
			hNextCtrl = GetDlgItem(hParent, table[i]);

			if(IsWindowEnabled(hNextCtrl) == 0) continue;

			SetFocus(hNextCtrl);
			break;
		}

		return;
	}


	nCtrlId = GetWindowLongPtr(hFocusedCtrl, GWLP_ID);


	//	Find position of a focused control in the table.
	//
	for(i = 0; i < num; i++)
	{
		if(nCtrlId == table[i])
		{
			nCtrlPos = i;
			break;
		}
	}

	if(nCtrlPos == -1)
	{
		IDE_Warn("Can't find focused control in focus rotation table.");
		return;
	}


	//	Move focus on the next non-disabled control.
	//
	for(i = 0; i < num; i++)
	{
		nCtrlPos++;

		//	If the focus was on the last control, move it to the first control
		//	from the table.
		//
		if(nCtrlPos >= num) nCtrlPos = 0;

		hNextCtrl = GetDlgItem(hParent, table[nCtrlPos]);
		if(IsWindowEnabled(hNextCtrl) == 0) continue;
		SetFocus(hNextCtrl);
		break;
	}
}






//	Save information about current state of an EDIT control.
//
void IDE_SaveCtrlEditState(HWND hEdit, IDE_CTRL_EDIT_STATE *table, int rows)
{
	LONG id;
	int i;


	id = GetWindowLong(hEdit, GWL_ID);

	for(i = 0; i < rows; i++)
	{
		if(id == table[i].id) break;
	}

	if(i >= rows)
	{
		IDE_Warn("Can't find specified EDIT control in state table for saving.");
		return;
	}

	GetWindowText(hEdit, table[i].text, sizeof(table[i].text));
	SendMessage(hEdit, EM_GETSEL, (WPARAM)&table[i].selStart, (LPARAM)&table[i].selEnd);
}




//	Load information about previous state of an EDIT control.
//
void IDE_LoadCtrlEditState(HWND hEdit, IDE_CTRL_EDIT_STATE *table, int rows)
{
	LONG id;
	int i;
	char text[IDE_CTRL_EDIT_MAXLEN];
	DWORD selStart, selEnd;

	
	id = GetWindowLong(hEdit, GWL_ID);

	for(i = 0; i < rows; i++)
	{
		if(id == table[i].id) break;
	}

	if(i >= rows)
	{
		IDE_Warn("Can't find an EDIT control in state table for loading.");
		return;
	}

	//	Save current state of the control for futher rotation.
	//
	GetWindowText(hEdit, text, sizeof(text));
	SendMessage(hEdit, EM_GETSEL, (WPARAM)&selStart, (LPARAM)&selEnd);

	//	Apply state stored in table on the EDIT control.
	//
	SetWindowText(hEdit, table[i].text);
	SendMessage(hEdit, EM_SETSEL, (WPARAM)table[i].selStart, (LPARAM)table[i].selEnd);

	//	Save previous state of the control to the table.
	//
	strcpy(table[i].text, text);
	table[i].selStart = selStart;
	table[i].selEnd = selEnd;
}




//	Remove information stored in control state table. This function is called
//	when an user closes a window containing any EDIT controls.
//
void IDE_PurgeCtrlStateTable(IDE_CTRL_EDIT_STATE *table, int rows)
{
	int i;

	for(i = 0; i < rows; i++)
	{
		strcpy(table[i].text, "");
		table[i].selStart = 0;
		table[i].selEnd = 0;
	}
}








//	Copy text to clipboard of operating system. This function returns TRUE on
//	success and FALSE on failure.
//
BOOL IDE_CopyTextToClipboard(char *text)
{
	HGLOBAL hMem;
	int len;
	char *addr;

	if(OpenClipboard(IDE_hWndMain) == 0)
	{
		IDE_Warn("Can't open clipboard for copying a text.");
		return FALSE;
	}

	if(EmptyClipboard() == 0)
	{
		CloseClipboard();
		IDE_Warn("Can't purge clipboard before copying a text.");
		return FALSE;
	}

	//	Allocate global block of memory from the heap for the text.
	//
	len = strlen(text);

	hMem = GlobalAlloc(GMEM_MOVEABLE, len + 1);
	if(hMem == NULL)
	{
		CloseClipboard();
		IDE_Warn("Can't allocate memory for copying a text into clipboard.");
		return FALSE;
	}

	//	Lock the block of memory from moving and discarding, copy text into
	//	clipboard and unlock the memory.
	//
	addr = GlobalLock(hMem);
	if(addr == NULL)
	{
		CloseClipboard();
		IDE_Warn("Can't lock memory before copying a text into clipboard.");
		return FALSE;
	}

	strcpy(addr, text);

	if(GlobalUnlock(hMem) == 0 && GetLastError() != NO_ERROR)
	{
		CloseClipboard();
		IDE_Warn("Can't unlock memory after copying a text for clipboard.");
		return FALSE;
	}

	//	Bound clipboard with block of memory containing the text.
	//
	if(SetClipboardData(CF_TEXT, hMem) == NULL)
	{
		CloseClipboard();
		IDE_Warn("Can't bound clipboard with block of memory with text.");
		return FALSE;
	}


	if(CloseClipboard() == 0) IDE_Warn("Can't close clipboard after copying text.");


	return TRUE;
}










HWND App_AddTooltip(HWND hParent, UINT childId, char *text)
{
	TOOLINFO toolInfo = {0};

	HWND hChild = GetDlgItem(hParent, childId);

	HWND hTooltip = CreateWindowEx
	(
		NULL, TOOLTIPS_CLASS, NULL,
		WS_POPUP | TTS_ALWAYSTIP,
		CW_USEDEFAULT, CW_USEDEFAULT,
		CW_USEDEFAULT, CW_USEDEFAULT,
		hParent, NULL,
		IDE_hInstance, NULL
	);

	// Associate the tooltip with the tool.
	//
	toolInfo.cbSize = sizeof(toolInfo);
	toolInfo.hwnd = hParent;
	toolInfo.uFlags = TTF_IDISHWND | TTF_SUBCLASS;
	toolInfo.uId = (UINT_PTR)hChild;
	toolInfo.lpszText = text;

	SendMessage(hTooltip, TTM_ADDTOOL, 0, (LPARAM)&toolInfo);



	//	HOW TO USE
	//

		//	Add tooltip.
		//
		//App_AddTooltip(hwndDlg, ID_DLG_FILE_NEWPROJECT_NAME, "tatssty/");


		//	MANUAL CONTROL.
		//
		//SendMessage(hwndTip, TTM_TRACKPOSITION, 0, MAKELPARAM(screenX, screenY));
		
		// show the tracking tooltip for the tool identified by TOOLINFO
		//SendMessage(hwndTip, TTM_TRACKACTIVATE, TRUE, (LPARAM)&toolInfo);
		
		// to hide it later:
		//SendMessage(hwndTip, TTM_TRACKACTIVATE, FALSE, (LPARAM)&toolInfo);







	return hTooltip;
}




































